//*************************************************************************************************
//
//	Description:
//		basic.fx - Basic object shader for The Race. Supports vertex colours, Blinn-style diffuse
//		and specular lighting, shininess maps, tangent space normal maps, multiple light sources
//		(of 3 different types), shadow buffers.
//
//	<P> Copyright (c) 2006 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Tom Nettleship
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		TNettleship     04/05/2006  0.1           Created
//		TNettleship     05/12/2006  1.0           First "offical" release to FTP
//		TNettleship     05/18/2006  1.1           Added technique annotations.
//		TNettleship     05/31/2006  1.2           Added renderer lighting interface.
//		TNettleship     05/31/2006  1.3           Added specular overlighting to game engine code.
//		TNettleship     06/08/2006  1.4           Added ambient occlusion term from vertex colour,
//																							global emap factor, added option for compressed
//																							normals.
//		TNettleship			08/01/2006	1.5						Added option for ambient occlusion to affect
//																							diffuse and specular lighting. Fixed a bug with
//																							channel assignment & vertex colours in 3DSMAX.
//		TNettleship			10/16/2006	1.6						Changed the way lighting distance attenuation works.
//		TNettleship			10/19/2006	1.7						Removed texture mapping info params from display,
//																							reverted to parser 0x0000 until the texture management
//																							bugs in 0x0001 can be fixed.
//		TNettleship			11/02/2006	1.8						Fixed a bug with directional lighting.
//		TNettleship			11/21/2006	1.9						Added support for deferred rendering.
//		TMann						11/23/2006	1.12					Added _WIN_GL_ support
//		TMann						11/27/2006	1.13					Texture filtering for GL
//		TNettleship			12/04/2006	1.14					Added premultiplied alpha code to the fragment shader,
//		TMann						12/11/2006	1.15					PS3 support
//																							added support for double sided rendering.
//		TNettleship			01/02/2007	1.16					Stopped premultiplied alpha from being used in opaque
//																							materials.
//		TMann						01/09/2007	1.17					Added PS3/GL literal param passing
//		TNettleship			01/31/2007	1.18					Pixel shader optimisations
//		TNettleship			02/06/2007	1.19					Fixed translucent_twosided technique
//		TNettleship			02/12/2007	1.20					Fixed environment mapping
//		BIrvine					02/13/2007	1.21					Temporary shadow implementation
//		TNettleship			04/18/2007	2.0						Added shader specialisation.
//		TNettleship			05/03/2007	2.01					Added shadow override technique for basic_translucent.
//		TNettleship			07/10/2007	2.02					Added support for vertex alpha, added alphatest technique.
//		TNettleship			07/10/2007	2.03					Changed lighting to work in world coords.
//		TNettleship     07/24/2007  2.04          Made sure samplers aren't using anisotropic filtering.
//		TNettleship     08/17/2007  2.05          Removed texture bias.
//		TNettleship			08/30/2007	2.06					Mods for atoc support.
//		TNettleship			09/25/2007	2.07					Removed obsolete techniques, changed fresnel to be an
//																							optional specialisation.
//		TNettleship			10/23/2007	2.08					Converted to on-load rendermode behaviour binding.
//	<TABLE>
//
//*************************************************************************************************
#ifdef WRAP_AROUND_LIGHTING
#define _USE_HALF_LAMBERTIAN_LIGHTING_
#endif

#define _SSAO_READY_

#include "stddefs.fxh"
#include "specialisation_globals.fxh"


//-----------------------------------------------------------------------
//
// Preprocessor definitions
//

// Uncomment this to switch to "lets see where the envmaps are used" mode
//#define ENVMAP_SHOW_AND_TELL

#define MAX_ENVMAP_BIAS	6.0f

// Uncomment to use experimental higher quality envmapping approach
//#define EXPERIMENTAL_ENVMAPPING

// Compiler test settings, exercises all options
#if defined( TEST_COMPILE )
//#define USE_ANISO
#define USE_SPECULAR
#define NORMAL_MAPPING
#define ENV_MAPPING
//#define USE_FRESNEL
//#define USE_LIGHT_CONTROL
#endif


//-----------------------------------------------------------------------
//
// Input parameters
//

//
// Camera
//
#ifdef _3DSMAX_
// 3DSMax parser 0x0001 doesn't support WorldCameraPosition, so we need to bring the view matrix
// in to access the 4th row to get the same information. Parser 0x0000 supports it. Bleh.
float4x4 viewI : ViewInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
// The ingame renderer directly supplies the camera position
SHARE_PARAM float3 worldCameraPos : WorldCameraPosition
<
	string UIWidget = "None";
	bool appEdit = false;
>;
#endif



//
// Transforms
//
#if defined( _3DSMAX_ ) || defined(USE_WVP_CONSTANT)
// Max doesn't support viewproj as an app-supplied parameter
float4x4 worldviewproj : WorldViewProjection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
SHARE_PARAM float4x4 viewproj : ViewProjection
<
	bool appEdit = false;
	bool export = false;
>;
#endif

float4x4 world : World
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;



//
// Channel mappings (max only)
//

//
// N.B. Max contains a bug which means the colour channel must NOT be mapped to texcoord0.
// The first UV coord channel MUST be mapped to texcoord0 or the basis vectors for normal
// mapping will be screwed up. (e.g. there's some bit of code deep within max which assumes
// this setup when calculating the basis vectors)
//

#ifdef _3DSMAX_

// First UV channel
int texcoord0 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 0;
	int MapChannel = 1;
	int RuntimeTexcoord = 0;
	bool export = false;
> = 0;

// Vertex colour channel
int texcoord1 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 1;
	int MapChannel = 0;
	bool ColorChannel = true;
	bool export = false;
> = 0;

// Vertex alpha channel (max presents it seperately for no good reason)
int texcoord2 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 2;
	int MapChannel = -2;
	bool ColorChannel = true;
	bool export = false;
>;

// Lighting control channel, uses map channel 2
int texcoord3 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 3;
	int MapChannel = 2;
	int RuntimeTexcoord = 1;
	bool ProtectFromRescaling = true;
	bool ColorChannel = true;
	bool export = false;
> = 0;

#endif


//
// Textures
//

SPECIALISATION_PARAM( useAnisotropic, "Anisotropic Filter?", "USE_ANISO" )	// TRUE if the anisotropic filtering is to be used

#ifdef _3DSMAX_
texture diffuseTexture : DiffuseMap						// Diffuse colour in RGB, translucency in alpha
#else
texture diffuseTexture : TEXTURE							// Diffuse colour in RGB, translucency in alpha
#endif
<
	string UIName = "Diffuse Tex {UV1}";
	bool appEdit = true;
>;

SPECIALISATION_PARAM_DEFAULTS_TRUE( useSpecular, "Use specular?", "USE_SPECULAR" )	// TRUE if the specular lighting is to be used

#if defined( _3DSMAX_ ) || defined( USE_SPECULAR )
DEPENDENT_TEXTURE_PARAM( specularTexture, "Spec Tex {UV1}", useSpecular )
#endif

SPECIALISATION_PARAM( useNormalMap, "Use normal map?", "NORMAL_MAPPING" )	// TRUE if the normal map is to be used in lighting
DECLARE_DEPENDENT_VERTEX_STREAM( tangentDependency, tangent, TANGENT, useNormalMap )
DECLARE_DEPENDENT_VERTEX_STREAM( binormalDependency, binormal, BINORMAL, useNormalMap )

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
DEPENDENT_TEXTURE_PARAM( normalTexture, "Normal Tex {UV1}", useNormalMap )
DEPENDENT_BOOL_PARAM( specialNormalMap, "Special normal map format?", useNormalMap )
#endif

SPECIALISATION_PARAM( useEnvironmentMap, "Use environment map?", "ENV_MAPPING" )	// TRUE if the environment map is to be used

#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
DEPENDENT_CUBE_TEXTURE_PARAM( environmentTexture, "Env Texture", useEnvironmentMap, false )
#endif

SPECIALISATION_PARAM( useFresnel, "Use Fresnel?", "USE_FRESNEL" )	// TRUE if fresnel factor should be used

#if defined( _3DSMAX_ ) || defined( USE_FRESNEL )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( fresnelFactor, "Fresnel Factor", useFresnel, 0.0f, 1.0f, 0.6f )
#endif

#if defined( _3DSMAX_ ) || defined( USE_SPECULAR )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( minSpecPower, "Min Specular Power", useSpecular, 1.0f, 1024.0f, 1.0f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( maxSpecPower, "Max Specular Power", useSpecular, 1.0f, 1024.0f, 32.0f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( globalSpecularFactor, "Specular Factor", useSpecular, 0.0f, 10.0f, 1.0f )
#endif

#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( globalEMapFactor, "EMap Factor", useEnvironmentMap, 0.0f, 1.0f, 1.0f )
#endif

SPECIALISATION_PARAM( wrapAround, "Lighting Wrap Around", "WRAP_AROUND_LIGHTING" )	// only works in game, cannot be done in Max 

//
// Lighting
//

#include "lighting_globals.fxh"
DECLARE_LIGHTING_PARAMS

// colour multiplier, forced to end to avoid artists touching it

float4 globalColourMultiplier
<
	string UIwidget = "None";
	bool appEdit = true;
	bool export = true;
> = { 1.0f, 1.0f, 1.0f, 1.0f };


// TRUE if the material represents vehicle lights & so self-emits under code control
SPECIALISATION_PARAM( useLightControl, "Light Control?", "USE_LIGHT_CONTROL" )
DECLARE_DEPENDENT_VERTEX_STREAM( texcoord2Dependency, texCoord2, TEXCOORD3, useLightControl )

#if defined( _3DSMAX_ ) || defined( USE_LIGHT_CONTROL )
DEPENDENT_TEXTURE_PARAM( emissiveTexture, "Emissive Tex{UV1}", useLightControl )

#define NUM_EMISSION_CONTROL_VALUES		8
float emissionControl[ NUM_EMISSION_CONTROL_VALUES ]
<
	string UIName = "emissionControl";
	bool appEdit = true;
	bool export = false;
	string exportDependsOn = "useLightControl";
>;

float testLightLevel
<
	string UIName = "Test Light Level";
	bool appEdit = false;
	bool export = false;
> = 0.0f;
#endif



//-----------------------------------------------------------------------
//
// Samplers
//

#if defined( USE_ANISO )

sampler2D diffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="diffuseTexture"; 
	string MinFilter = "Anisotropic";
	string MagFilter = "Anisotropic";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
	int MaxAnisotropy = 6;
> 
= sampler_state
{
	Texture = < diffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _ANISOMINFILTER;
	MagFilter = _ANISOMAXFILTER;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_MAX_ANISOTROPY( 6 )
#endif
};

#else

sampler2D diffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="diffuseTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < diffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

#endif

#if defined( _3DSMAX_ ) || defined( USE_SPECULAR )
sampler2D specularMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="specularTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < specularTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
sampler2D normalMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="normalTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < normalTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif


#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
samplerCUBE environmentMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="environmentTexture";
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
	string AddressW  = "Clamp";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < environmentTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
#if defined(_PS3_)
	AddressU  = ClampToEdge;
	AddressV  = ClampToEdge;
	AddressW  = ClampToEdge;
	LODBias = 0;
#else
	AddressU  = Clamp;
	AddressV  = Clamp;
	AddressW  = Clamp;
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif


#if defined( _3DSMAX_ ) || defined ( USE_LIGHT_CONTROL )
sampler2D emissiveMap : SAMPLER 
<
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="emissiveTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < emissiveTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif


//
// Functions
//


//
// This seems to work almost as well as the full-on "complicated fresnel"
// A good rindexRatio for car paint is 0.6667 (which assumes air = 1.0, clearcoat = 1.5)
//
float CalculateFresnel( float3 _eye, float3 _normal, float rindexRatio )
{
	// Note: compute R0 on the CPU and provide as a
	// constant; it is more efficient than computing R0 in
	// the vertex shader. R0 is:
	// float const R0 = pow(1.0-refractionIndexRatio, 2.0)
	// / pow(1.0+refractionIndexRatio, 2.0);
	// light and normal are assumed to be normalized
	float R0 = pow( 1.0f - rindexRatio, 2.0f) / pow ( 1.0f + rindexRatio, 2.0f );

	return R0 + ( 1.0f - R0 ) * pow( 1.0f - saturate( dot( -_eye, _normal ) ), 5.0f );
}


// Function calculates emissiveness by looking up a value from an array which the app sets,
// indexed by the control ID passed. If a negative ID is passed, the emission is considered to
// be always on.
#if defined( _3DSMAX_ ) || defined( USE_LIGHT_CONTROL )
float CalcEmissiveness( float _ID, float _brightness )
{
	int intControlID = floor( _ID );

	if ( intControlID >= 0 )
	{
#if defined( _3DSMAX_ )
		return testLightLevel * abs( _brightness );
#else
		return emissionControl[ intControlID ] * abs( _brightness );
#endif
	}
	else
	{
		return 0.0f;
	}
}
#endif



//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

// Input structure
struct VSINPUT
{
	float3 position : POSITION;														// Object space position
#ifdef _3DSMAX_
	float3 colour   : TEXCOORD1;													// Vertex colour
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord - N.B. MAx requires that texcoord0 is a geometric channel
																												// as it implicitly uses that to calculate the tangent space coordinate frame.
	float3 alpha		: TEXCOORD2;													// Vertex alpha
	float2 texCoord2 : TEXCOORD3;													// UV channel 2 texture coord (emissive control)
#else
	float4 colour   : COLOR0;															// Vertex colour
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord
#if defined( USE_LIGHT_CONTROL )
	float2 texCoord2 : TEXCOORD1;													// UV channel 2 texture coord (emissive control)
#endif
#endif
	float3 normal   : NORMAL;															// Object space normal

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
	// These two vertex streams aren't needed without normal_mapping
	float3 tangent  : TANGENT;														// Object space tangent
	float3 binormal : BINORMAL;														// Object space normal
#endif
};


struct SHADOWGEN_VSINPUT
{
	float3 position : POSITION;														// Object space position
	float4 colour   : COLOR0;															// Vertex colour
	float2 texCoord : TEXCOORD0;													// Texture coords
};


// Output structure
struct VSOUTPUT
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD4;														// Vertex colour
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
	float4 normal_and_fresnel	: TEXCOORD1;								// Normal vector (world space) + fresnel coefficient
	float4 eye				: TEXCOORD2;												// Eye vector (world space)

#if defined( _3DSMAX_ ) || defined( USE_LIGHT_CONTROL )
	// This vertex stream isn't needed without emissive lighting
	float emissiveness : TEXCOORD3;												// Emissiveness amount
#endif

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
	// These two vertex streams aren't needed without normal_mapping
	float3 tangent		: TEXCOORD5;												// Tangent vector (world space)
	float3 binormal		: TEXCOORD6;												// Binormal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS_VS( 7 )
#else
	DECLARE_LIGHTING_INTERPOLATORS_VS( 5 )
#endif
};



struct SHADOWGEN_VSOUTPUT
{
	float4 position			: POSITION;
	float4 colour				: TEXCOORD2;
	float2 texCoord			: TEXCOORD0;
	float4 shadowCoord	: TEXCOORD1;
};

struct ZPRIMEDOF_VSOUTPUT
{
	float4 position			: POSITION;
	float4 colour				: TEXCOORD2;
	float2 texCoord			: TEXCOORD0;
	float4 coords			: TEXCOORD1;
};


//-----------------------------------------------------------------------
//
// Vertex shader code
//

VSOUTPUT BasicVertexShader( VSINPUT _input )
{
	VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	// Copy simple invariant params to output structure
#if defined( _3DSMAX_ )
	_output.colour.rgb = _input.colour;
	_output.colour.a = _input.alpha.r;
#else
	_output.colour = _input.colour;
#endif
	_output.texCoord = _input.texCoord;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;

	// Calculate world-space vector to the eye
#ifdef _3DSMAX_
	float3 worldEyeVec = viewI[ 3 ] - worldPos;
#else
	float3 worldEyeVec = worldCameraPos - worldPos;
#endif
	_output.eye = float4(worldEyeVec,0);

	float3 normal = normalize( mul( float4( _input.normal, 0.0f ), world ).xyz );

	DEPENDENT_CODE_START( useLightControl )
#if defined( _3DSMAX_ ) || defined( USE_LIGHT_CONTROL )
		_output.emissiveness = CalcEmissiveness( _input.texCoord2.x, _input.texCoord2.y );
#endif
	DEPENDENT_CODE_END( useLightControl )

	DEPENDENT_CODE_START( useNormalMap )
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
		_output.tangent  = mul( float4( _input.tangent, 0.0f ), world ).xyz;
		_output.binormal = mul( float4( _input.binormal, 0.0f ), world ).xyz;
#endif
	DEPENDENT_CODE_END( useNormalMap )

	// If its required
	DEPENDENT_CODE_START( useFresnel )
#if defined( _3DSMAX_ ) || defined( USE_FRESNEL )
		// Calculate fresnel term
		_output.normal_and_fresnel = float4( normal, CalculateFresnel( - normalize( worldEyeVec ), normal, fresnelFactor ) );
#endif
	DEPENDENT_CODE_ELSE( useFresnel )
#if defined( _3DSMAX_ ) || !defined( USE_FRESNEL )
		// Fresnel term is 1.0
		_output.normal_and_fresnel = float4( normal, 1.0f );
#endif
	DEPENDENT_CODE_END( useFresnel )

#if !defined( _3DSMAX_ ) && !defined( USE_SPECULAR )
	// In the engine, specialisations which have no specular defined need to declare these constants
	// as the lighting macros at the end need them, and they're not declared anywhere else.
	float globalSpecularFactorValue = 0.0f;
	float minSpecPowerValue = 1.0f;
	float maxSpecPowerValue = 1.0f;
#else
	DEPENDENT_CODE_START( useSpecular )
	float globalSpecularFactorValue = globalSpecularFactor;
	DEPENDENT_CODE_END( useSpecular )
#endif

	// Do lighting
	DO_VS_LIGHTING_CALCULATIONS

	_output.colour *= globalColourMultiplier;

	return _output;
}



SHADOWGEN_VSOUTPUT BasicTranslucentShadowGenVertexShader( SHADOWGEN_VSINPUT _input )
{
	SHADOWGEN_VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	_output.texCoord = _input.texCoord;
	_output.colour	 = _input.colour;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// Clamp geometry that is behind the camera to the near plane so that it still renders.
	// We use an orthogonal projection, so doing this will not distort the shadow caster.
	CLAMP_SHADOW_Z;

	OUTPUT_SHADOW_COORDS;

	return _output;
}


ZPRIMEDOF_VSOUTPUT BasicTranslucentZPrimeDOFVertexShader( SHADOWGEN_VSINPUT _input )
{
	ZPRIMEDOF_VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	_output.texCoord = _input.texCoord;
	_output.colour	 = _input.colour;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );
	_output.coords = _output.position;

	return _output;
}



//-----------------------------------------------------------------------
//
// Fragment Shader(s)
//

// Input structure

#if defined( _3DSMAX_ )
// Max can't handle centroid interpolators properly

struct PSINPUT
{
	float4 colour			: TEXCOORD4;								// Vertex colour
	float2 texCoord		: TEXCOORD0;						// UV coords for texture channel 0
	float4 normal_and_fresnel	: TEXCOORD1;		// Normal vector (world space) + fresnel coefficient
	float4 eye				: TEXCOORD2;						// Eye vector (world space)
	float emissiveness : TEXCOORD3;						// Emissiveness amount
	float3 tangent		: TEXCOORD5;						// Tangent vector (world space)
	float3 binormal		: TEXCOORD6;						// Binormal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS_PS( 7 )
};

#else

// Input structure
struct PSINPUT
{
	float4 colour			: TEXCOORD4_centroid;											// Vertex colour
	float2 texCoord		: TEXCOORD0;									// UV coords for texture channel 0
	float4 normal_and_fresnel	: TEXCOORD1_centroid;	// Normal vector (world space) + fresnel coefficient
	float4 eye				: TEXCOORD2_centroid;					// Eye vector (world space)

#if defined( _3DSMAX_ ) || defined( USE_LIGHT_CONTROL )
	// This vertex stream isn't needed without emissive lighting
	float emissiveness : TEXCOORD3;									// Emissiveness amount
#endif

#if defined( NORMAL_MAPPING )
	// These two vertex streams aren't needed without normal_mapping
	float3 tangent		: TEXCOORD5_centroid;					// Tangent vector (world space)
	float3 binormal		: TEXCOORD6_centroid;					// Binormal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS_PS( 7 )
#else
	DECLARE_LIGHTING_INTERPOLATORS_PS( 5 )
#endif
	DECLARE_SHADOW_PS_INPUTS
};

#endif	// defined( _3DSMAX_ )


struct SHADOWGEN_PSINPUT
{
	float4 colour				: TEXCOORD2;
	float2 texCoord			: TEXCOORD0;
	float4 shadowCoord	: TEXCOORD1;
};


// Output structure
struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE Colour : COLOR0;
};



//-----------------------------------------------------------------------
//
// Fragment shader code
//

REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT BasicFragmentShader( PSINPUT _input )
{
	PSOUTPUT _output;

	PS_GENERATE_WORLDPOS( _input.eye.xyz )

	// Read textures
	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );
	float4 specularTexColour;
#if defined( _3DSMAX_ ) || defined( USE_LIGHT_CONTROL )
	float4 emissiveColour = tex2D( emissiveMap, _input.texCoord );
#endif

	float globalSpecularFactorValue;
	float minSpecPowerValue;
	float maxSpecPowerValue;

#if !defined( _3DSMAX_ ) && !defined( USE_SPECULAR )
	// In the engine, specialisations which have no specular defined need to declare these constants
	// as the lighting macros at the end need them, and they're not declared anywhere else.
#else
	DEPENDENT_CODE_START( useSpecular )
	globalSpecularFactorValue = globalSpecularFactor;
	minSpecPowerValue = minSpecPower;
	maxSpecPowerValue = maxSpecPower;
	DEPENDENT_CODE_END( useSpecular )
#endif

	DEPENDENT_CODE_START( useSpecular )
#if defined( _3DSMAX_ ) || defined( USE_SPECULAR )
		// Read specular texture
		specularTexColour = tex2D( specularMap, _input.texCoord );

		DEPENDENT_CODE_START( useEnvironmentMap )
		DEPENDENT_CODE_END( useEnvironmentMap )

#endif
	DEPENDENT_CODE_ELSE( useSpecular )
#if defined( _3DSMAX_ ) || !defined( USE_SPECULAR )
		// No specular, so default the colour to ones (will be optimised out)
		specularTexColour = float4( 1.0f, 1.0f, 1.0f, 0.0f );

		globalSpecularFactorValue = 0.0f;
		minSpecPowerValue = 0.0f;
		maxSpecPowerValue = 0.0f;
#endif
	DEPENDENT_CODE_END( useSpecular )

	// Factor vertex alpha into the diffuse alpha
	diffuseTexColour.a *= _input.colour.a;

  // Normalise interpolated vectors
	float3 TSnormal = normalize( _input.normal_and_fresnel.xyz );
  float3 eye = normalize( _input.eye.xyz );
	float3 normal;

	// Do tangent space normal mapping if required
	DEPENDENT_CODE_START( useNormalMap )
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
		// Normalise the input tangent and binormal vectors
		float3 tangent = normalize( _input.tangent );
		float3 binormal = normalize( _input.binormal );

		// Fetch and decode the map normal
		float4 normalMapColour = tex2D( normalMap, _input.texCoord );

		float3 normalFromMap;
		if ( specialNormalMap )
		{
			normalFromMap = normalize( ( normalMapColour.agb * 2.0f ) - 1.0f );
		}
		else
		{
			normalFromMap = normalize( ( normalMapColour.rgb * 2.0f ) - 1.0f );
		}

		// Perturb the tangent space normal by the normal map
		normal = ( TSnormal * normalFromMap.z ) + ( normalFromMap.x * binormal ) + ( normalFromMap.y * tangent );
		normal = normalize( normal );
#endif
	DEPENDENT_CODE_ELSE( useNormalMap )
#if defined( _3DSMAX_ ) || !defined( NORMAL_MAPPING )
		// No normal map, so use interpolated normal
		normal = TSnormal;
#endif
	DEPENDENT_CODE_END( useNormalMap )

	// Calculate base colour
	float4 accumulatedColour = diffuseTexColour * _input.colour;

	DEPENDENT_CODE_START( useLightControl )
#if defined( _3DSMAX_ ) || defined( USE_LIGHT_CONTROL )
	// Add in emissive lighting
	accumulatedColour.rgb += _input.emissiveness * emissiveColour.rgb * emissiveColour.a;
#endif
	DEPENDENT_CODE_END( useLightControl )

	//
	// Calculate the fresnel factor
	//
	float fresnelCoefficient;
	DEPENDENT_CODE_START( useFresnel )
#if defined( _3DSMAX_ ) || defined( USE_FRESNEL )
		fresnelCoefficient = saturate( _input.normal_and_fresnel.w );
#endif
	DEPENDENT_CODE_ELSE( useFresnel )
#if defined( _3DSMAX_ ) || !defined( USE_FRESNEL )
		fresnelCoefficient = 1.0f;
#endif
	DEPENDENT_CODE_END( useFresnel )

	// If environment mapping is switched on
	DEPENDENT_CODE_START( useEnvironmentMap )
#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
		float3 reflectionVector = reflect( -eye, normal );

#if defined( EXPERIMENTAL_ENVMAPPING )
	// Intersect the reflection vector with a 100m sphere centred on the object
	float3 rayOrigin = worldPos - world[3].xyz;
	float3 rayDirection = reflectionVector;
	float B = dot( rayDirection, rayOrigin ) * 2.0f;
	float C = dot( rayOrigin, rayOrigin ) - ( 100.0f * 100.0f );
	float t0 = -B - sqrt( B*B - 4.0f * C ) / 2.0f;
	float t1 = -B + sqrt( B*B - 4.0f * C ) / 2.0f;
	float3 sphereIntersection = rayOrigin + ( rayDirection * max( t0, t1 ) );

	// Use the intersection point to look up the cubemap, not the reflection vector
	reflectionVector = normalize( sphereIntersection );
#endif

		// Fetch the environment map colour using the world coords vector
#ifdef _3DSMAX_
		float4 environmentTexColour = texCUBElod( environmentMap, float4( reflectionVector.xzy, MAX_ENVMAP_BIAS * ( 1.0f - specularTexColour.a ) ) );
#else
		float4 environmentTexColour = texCUBElod( environmentMap, float4( reflectionVector, MAX_ENVMAP_BIAS * ( 1.0f - specularTexColour.a ) ) );
#endif

#if defined( ENVMAP_SHOW_AND_TELL )
		accumulatedColour = environmentTexColour;
#else
		// Calculate envmap colour and add to diffuse
		accumulatedColour += specularTexColour * environmentTexColour * globalEMapFactor * fresnelCoefficient;
#endif

#endif
	DEPENDENT_CODE_END( useEnvironmentMap )

#if !defined( ENV_MAPPING ) || !defined( ENVMAP_SHOW_AND_TELL )

	// Perform lighting
	DO_PS_LIGHTING_CALCULATIONS( accumulatedColour , _input.eye.xyz )

#endif

	accumulatedColour.w = diffuseTexColour.w;
	_output.Colour = CalculateOutputPixel( accumulatedColour );


	return _output;
}


//
// Low Detail Shaders
//


struct VSOUTPUT_LD
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD1;														// Vertex colour
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
#if defined( _3DSMAX_ ) || defined( USE_LIGHT_CONTROL )
	// This vertex stream isn't needed without emissive lighting
	float emissiveness : TEXCOORD2;										// Emissiveness amount
#endif
};


VSOUTPUT_LD BasicLowDetailVertexShader( VSINPUT _input )
{
	VSOUTPUT_LD _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	_output.texCoord = _input.texCoord;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// get normal in world space and do lighting
	float3 normal = normalize( mul( float4( _input.normal, 0.0f ), world ).xyz );
	
	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;

	DEPENDENT_CODE_START( useLightControl )
#if defined( _3DSMAX_ ) || defined( USE_LIGHT_CONTROL )
		_output.emissiveness = CalcEmissiveness( _input.texCoord2.x, _input.texCoord2.y );
#endif
	DEPENDENT_CODE_END( useLightControl )

#if defined( _3DSMAX_ )
	_output.colour.rgb = _input.colour;
	_output.colour.a = _input.alpha.r;
#else
	_output.colour = _input.colour;
#endif
	DO_VERTEX_LIGHTING( worldPos, normal, _output.colour )

	// keep alpha value from the vertex
#if defined( _3DSMAX_ )
	_output.colour.a = _input.alpha.r;
#else
	_output.colour.a = _input.colour.a;
#endif

	return _output;
}

REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT BasicLowDetailFragmentShader( VSOUTPUT_LD _input )
{
	PSOUTPUT _output;

	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );

	// Factor vertex alpha into the diffuse alpha
	diffuseTexColour.a *= _input.colour.a;

	// Calculate base colour
	float4 accumulatedColour = diffuseTexColour * _input.colour;

	DEPENDENT_CODE_START( useLightControl )
#if defined( _3DSMAX_ ) || defined( USE_LIGHT_CONTROL )
	// Add in emissive lighting
	float4 emissiveColour = tex2D( emissiveMap, _input.texCoord );
	accumulatedColour.rgb += _input.emissiveness * emissiveColour.rgb * emissiveColour.a;
#endif
	DEPENDENT_CODE_END( useLightControl )

	accumulatedColour.w = diffuseTexColour.w;
	_output.Colour = CalculateLowDetailOutputPixel( accumulatedColour );

	return _output;
}

REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT BasicTranslucentShadowGenFragmentShader( SHADOWGEN_PSINPUT _input )
{
	PSOUTPUT output;

	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );

	CALC_SHADOWMAP_DEPTH( output.Colour, _input.shadowCoord );
	output.Colour.a = diffuseTexColour.a * _input.colour.a;
	SHADOWMAP_PS_ALPHATEST( output.Colour.a, 0.25f );

	return output;
}

REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT BasicTranslucentZPrimeDOFFragmentShader( ZPRIMEDOF_VSOUTPUT _input )
{
	PSOUTPUT output;

	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );

	output.Colour = _input.coords.z / _input.coords.w;
	output.Colour.a = diffuseTexColour.a * _input.colour.a;

	return output;
}



//-----------------------------------------------------------------------
//
// Technique(s)
//

technique Basic
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Basic";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_RENDER_DEFAULT";
	string zprimeDOFBehaviour	= "ERMB_RENDER_DEFAULT";
	string shadowGenBehaviour = "ERMB_RENDER_DEFAULT";
	string lowDetailBehaviour	= "ERMB_RENDER";
	string lowDetailTechnique	= "Basic_LowDetail";
	int    lowDetailDeferredID = 0;
	bool   appCanOverrideSampler = true;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx BasicVertexShader();
		PixelShader = compile sce_fp_rsx BasicFragmentShader();
#else
		VertexShader = compile vs_3_0 BasicVertexShader();
		PixelShader = compile ps_3_0 BasicFragmentShader();
#endif
	}
}



technique Basic_Translucent
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	string normalBehaviour			= "ERMB_RENDER";
	string normalTechnique			= "Basic";
	int    normalDeferredID			= 2;
	string zprimeBehaviour			= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour		= "ERMB_DONT_RENDER";
	string shadowGenBehaviour		= "ERMB_RENDER";
	string shadowGenTechnique		= "_Basic_Translucent_ShadowGen";
	int    shadowGenDeferredID	= 0;
	string lowDetailBehaviour		= "ERMB_RENDER";
	string lowDetailTechnique		= "Basic_LowDetail";
	int    lowDetailDeferredID	= 2;
	bool   appCanOverrideSampler = true;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool	ZEnable = true;
		bool	ZWriteEnable = false;
		bool	AlphaBlendEnable = true;
		string SrcBlend = "SRCALPHA";
		string DestBlend = "INVSRCALPHA";
		string BlendOp = "ADD";
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = false;
		AlphaBlendEnable = true;
		SrcBlend = SRCALPHA;
		DestBlend = INVSRCALPHA;
		BlendOp = ADD;
#endif

#ifdef _3DSMAX_
		// The rendermode mapping table above maps rendering of this technique
		// to other techniques in all modes, so it doesn't need its own compiled
		// shaders, except in max.
		VertexShader = compile vs_3_0 BasicVertexShader();
		PixelShader = compile ps_3_0 BasicFragmentShader();
#endif
	}
}



//
// Two sided technique is now obsolete
//
technique Basic_TwoSided
<
	bool obsolete = true;
	string replacementTechnique = "Basic";
>
{
#ifdef _3DSMAX_
	pass Pass0
	{
		VertexShader = compile vs_3_0 BasicVertexShader();
		PixelShader = compile ps_3_0 BasicFragmentShader();
	}
#endif
}


//
// Two sided translucent technique is now obsolete
//
technique Basic_Translucent_TwoSided
<
	bool obsolete = true;
	string replacementTechnique = "Basic_Translucent";
>
{
#ifdef _3DSMAX_
	pass Pass0
	{
		VertexShader = compile vs_3_0 BasicVertexShader();
		PixelShader = compile ps_3_0 BasicFragmentShader();
	}
#endif
}



technique Basic_AlphaTest
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	string normalBehaviour			= "ERMB_RENDER";
	string normalTechnique			= "Basic";
	int    normalDeferredID			= 0;
	string zprimeBehaviour			= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour		= "ERMB_RENDER";
	string zprimeDOFTechnique		= "_Basic_Translucent_ZPrime_DOF";
	int    zprimeDOFDeferredID	= 0;
	string shadowGenBehaviour		= "ERMB_RENDER";
	string shadowGenTechnique		= "_Basic_Translucent_ShadowGen";
	int    shadowGenDeferredID	= 0;
	string lowDetailBehaviour		= "ERMB_RENDER";
	string lowDetailTechnique		= "Basic_LowDetail";
	int    lowDetailDeferredID	= 0;
	bool   appCanOverrideSampler = true;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool	ZEnable = true;
		bool	ZWriteEnable = true;
		bool	AlphaBlendEnable = false;
		bool	AlphaTestEnable = true;
		int		AlphaRef = 64;
		string AlphaFunc = "GreaterEqual";
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
		AlphaTestEnable = true;
		AlphaRef = 0x40;
		AlphaFunc = GreaterEqual;
#endif


#ifdef _3DSMAX_
		// The rendermode mapping table above maps rendering of this technique
		// to other techniques in all modes, so it doesn't need its own compiled
		// shaders, except in max.
		VertexShader = compile vs_3_0 BasicVertexShader();
		PixelShader = compile ps_3_0 BasicFragmentShader();
#endif
	}
}



technique Basic_LowDetail
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Basic_LowDetail";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_RENDER_DEFAULT";
	string zprimeDOFBehaviour	= "ERMB_RENDER_DEFAULT";
	string shadowGenBehaviour = "ERMB_RENDER_DEFAULT";
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx BasicLowDetailVertexShader();
		PixelShader = compile sce_fp_rsx BasicLowDetailFragmentShader();
#else
		VertexShader = compile vs_3_0 BasicLowDetailVertexShader();
		PixelShader = compile ps_3_0 BasicLowDetailFragmentShader();
#endif
	}
}


technique Basic_Translucent_LowDetail
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	string normalBehaviour			= "ERMB_RENDER";
	string normalTechnique			= "Basic_LowDetail";
	int    normalDeferredID			= 2;
	string zprimeBehaviour			= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour		= "ERMB_DONT_RENDER";
	string shadowGenBehaviour		= "ERMB_RENDER";
	string shadowGenTechnique		= "_Basic_Translucent_ShadowGen";
	int    shadowGenDeferredID	= 0;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool	ZEnable = true;
		bool	ZWriteEnable = false;
		bool	AlphaBlendEnable = true;
		string SrcBlend = "SRCALPHA";
		string DestBlend = "INVSRCALPHA";
		string BlendOp = "ADD";
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = false;
		AlphaBlendEnable = true;
		SrcBlend = SRCALPHA;
		DestBlend = INVSRCALPHA;
		BlendOp = ADD;
#endif

#ifdef _3DSMAX_
		// The rendermode mapping table above maps rendering of this technique
		// to other techniques in all modes, so it doesn't need its own compiled
		// shaders, except in max.
		VertexShader = compile vs_3_0 BasicVertexShader();
		PixelShader = compile ps_3_0 BasicFragmentShader();
#endif
	}
}


//
// Two sided technique is now obsolete
//
technique Basic_TwoSided_LowDetail
<
	bool obsolete = true;
	string replacementTechnique = "Basic_LowDetail";
>
{
#ifdef _3DSMAX_
	pass Pass0
	{
		VertexShader = compile vs_3_0 BasicVertexShader();
		PixelShader = compile ps_3_0 BasicFragmentShader();
	}
#endif
}


//
// Two sided translucent technique is now obsolete
//
technique Basic_Translucent_TwoSided_LowDetail
<
	bool obsolete = true;
	string replacementTechnique = "Basic_Translucent_LowDetail";
>
{
#ifdef _3DSMAX_
	pass Pass0
	{
		VertexShader = compile vs_3_0 BasicVertexShader();
		PixelShader = compile ps_3_0 BasicFragmentShader();
	}
#endif
}


//
// This technique only differs from Basic by the renderstates
// it requires, so its a max-only technique.
//
technique Basic_AlphaTest_LowDetail
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	string normalBehaviour			= "ERMB_RENDER";
	string normalTechnique			= "Basic_LowDetail";
	int    normalDeferredID			= 0;
	string zprimeBehaviour			= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour		= "ERMB_RENDER";
	string zprimeDOFTechnique		= "_Basic_Translucent_ZPrime_DOF";
	int    zprimeDOFDeferredID	= 0;
	string shadowGenBehaviour		= "ERMB_RENDER";
	string shadowGenTechnique		= "_Basic_Translucent_ShadowGen";
	int    shadowGenDeferredID	= 0;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool	ZEnable = true;
		bool	ZWriteEnable = true;
		bool	AlphaBlendEnable = false;
		bool	AlphaTestEnable = true;
		int		AlphaRef = 64;
		string AlphaFunc = "GreaterEqual";
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
		AlphaTestEnable = true;
		AlphaRef = 0x40;
		AlphaFunc = GreaterEqual;
#endif

#ifdef _3DSMAX_
		// The rendermode mapping table above maps rendering of this technique
		// to other techniques in all modes, so it doesn't need its own compiled
		// shaders, except in max.
		VertexShader = compile vs_3_0 BasicVertexShader();
		PixelShader = compile ps_3_0 BasicFragmentShader();
#endif
	}
}



technique _Basic_Translucent_ShadowGen
{
	pass Pass0
	{
			SHADOWMAP_STATES_ALPHATEST( 0x40 )
#if defined (_PS3_)
			ZEnable = true;
			ZWriteEnable = true;
			VertexShader = compile sce_vp_rsx BasicTranslucentShadowGenVertexShader();
			PixelShader = compile sce_fp_rsx BasicTranslucentShadowGenFragmentShader();
#elif defined(_XBOX)
			VertexShader = compile vs_3_0 BasicTranslucentShadowGenVertexShader();
			PixelShader = compile ps_3_0 BasicTranslucentShadowGenFragmentShader();
#else
			ZEnable = true;
			ZWriteEnable = true;
			VertexShader = compile vs_3_0 BasicTranslucentShadowGenVertexShader();
			PixelShader = compile ps_3_0 BasicTranslucentShadowGenFragmentShader();
#endif
	}
}

technique _Basic_Translucent_ZPrime_DOF
{
	pass Pass0
	{
#if defined (_PS3_)
			AlphaTestEnable = true;
		  AlphaFunc = int2(GEqual, 0x40);
			ZEnable = true;
			ZWriteEnable = true;
			VertexShader = compile sce_vp_rsx BasicTranslucentZPrimeDOFVertexShader();
			PixelShader = compile sce_fp_rsx BasicTranslucentZPrimeDOFFragmentShader();
#else
			AlphaTestEnable = true;
			AlphaRef = 0x40;
		  AlphaFunc = GreaterEqual;
			ZEnable = true;
			ZWriteEnable = true;
			VertexShader = compile vs_3_0 BasicTranslucentZPrimeDOFVertexShader();
			PixelShader = compile ps_3_0 BasicTranslucentZPrimeDOFFragmentShader();
#endif
	}
}

technique Default
<
	bool supportsSpecialisedLighting = false;
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Default";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour	= "ERMB_DONT_RENDER";
	string shadowGenBehaviour = "ERMB_DONT_RENDER";
>
{
	pass Pass0
	{
#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx BasicVertexShader();
		PixelShader = compile sce_fp_rsx BasicFragmentShader();
#else
		VertexShader = compile vs_3_0 BasicVertexShader();
		PixelShader = compile ps_3_0 BasicFragmentShader();
#endif
	}
}
